#include <GL/glew.h>
#include <GLFW/glfw3.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <cmath>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

std::string loadShaderSource(const std::string& path) {
    std::ifstream file(path);
    std::stringstream ss;
    ss << file.rdbuf();
    return ss.str();
}

GLuint compileShader(const std::string& src, GLenum type) {
    GLuint s = glCreateShader(type);
    const char* c = src.c_str();
    glShaderSource(s, 1, &c, nullptr);
    glCompileShader(s);

    GLint success;
    glGetShaderiv(s, GL_COMPILE_STATUS, &success);
    if (!success) {
        char info[512];
        glGetShaderInfoLog(s, 512, nullptr, info);
        std::cout << "Shader compilation error:\n" << info << std::endl;
        std::exit(EXIT_FAILURE);
    }

    return s;
}

GLuint createProgram(const std::string& v, const std::string& f) {
    GLuint vs = compileShader(loadShaderSource(v), GL_VERTEX_SHADER);
    GLuint fs = compileShader(loadShaderSource(f), GL_FRAGMENT_SHADER);

    GLuint p = glCreateProgram();
    glAttachShader(p, vs);
    glAttachShader(p, fs);
    glLinkProgram(p);

    GLint success;
    glGetProgramiv(p, GL_LINK_STATUS, &success);
    if (!success) {
        char info[512];
        glGetProgramInfoLog(p, 512, nullptr, info);
        std::cout << "Program linking error:\n" << info << std::endl;
        std::exit(EXIT_FAILURE);
    }

    glDeleteShader(vs);
    glDeleteShader(fs);

    return p;
}
struct Instance {
    glm::vec4 pos_scale;
};

std::vector<Instance> instances = {
};
void mienger(int depth, Instance parent, int maxDepth) {
    if(depth == maxDepth){
        instances.push_back(parent);
        return;
    }

    float childSize = parent.pos_scale.w / 3.0f;

    for(int x=0; x<3; x++){
        for(int y=0; y<3; y++){
            for(int z=0; z<3; z++){
if ( (x==1 && y==1) || (x==1 && z==1) || (y==1 && z==1) ) continue;


                //int countCenter = (x==1) + (y==1) + (z==1); chicken wing

                //if(countCenter == 3) continue;
                //if(countCenter == 2) continue;
                //if(countCenter == 0) continue;

                glm::vec3 offset = glm::vec3(x-1, y-1, z-1) * childSize;
                glm::vec3 childCenter = glm::vec3(parent.pos_scale) + offset;
                Instance child{glm::vec4(childCenter, childSize)};
                mienger(depth+1, child, maxDepth);
            }
        }
    }
}


int main() {
    glfwSwapInterval(1);

    mienger(0,{glm::vec4(0,0,0,300)},4);
    std::cout<<instances.size()<<std::endl;
    glfwInit();
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 4);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
    

    GLFWwindow* window = glfwCreateWindow(800, 600, "Mienger", nullptr, nullptr);
    glfwMakeContextCurrent(window);
    glewInit();

    glEnable(GL_DEPTH_TEST);
    std::cout << "OpenGL version: " << glGetString(GL_VERSION) << std::endl;
    std::cout << "OpenGL version: " << glGetString(GL_VERSION) << std::endl;
    std::cout << "GLSL version: " << glGetString(GL_SHADING_LANGUAGE_VERSION) << std::endl;

    float vertices[] = {
        // positions          // normals
        // front face (z = 0.5)
        -0.5f, -0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
         0.5f, -0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
         0.5f,  0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
         0.5f,  0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
        -0.5f,  0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
        -0.5f, -0.5f,  0.5f,  0.0f, 0.0f, 1.0f,
    
        // back face (z = -0.5)
        -0.5f, -0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
         0.5f, -0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
         0.5f,  0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
         0.5f,  0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
        -0.5f,  0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
        -0.5f, -0.5f, -0.5f,  0.0f, 0.0f, -1.0f,
    
        // left face (x = -0.5)
        -0.5f,  0.5f,  0.5f, -1.0f, 0.0f, 0.0f,
        -0.5f,  0.5f, -0.5f, -1.0f, 0.0f, 0.0f,
        -0.5f, -0.5f, -0.5f, -1.0f, 0.0f, 0.0f,
        -0.5f, -0.5f, -0.5f, -1.0f, 0.0f, 0.0f,
        -0.5f, -0.5f,  0.5f, -1.0f, 0.0f, 0.0f,
        -0.5f,  0.5f,  0.5f, -1.0f, 0.0f, 0.0f,
    
        // right face (x = 0.5)
         0.5f,  0.5f,  0.5f,  1.0f, 0.0f, 0.0f,
         0.5f,  0.5f, -0.5f,  1.0f, 0.0f, 0.0f,
         0.5f, -0.5f, -0.5f,  1.0f, 0.0f, 0.0f,
         0.5f, -0.5f, -0.5f,  1.0f, 0.0f, 0.0f,
         0.5f, -0.5f,  0.5f,  1.0f, 0.0f, 0.0f,
         0.5f,  0.5f,  0.5f,  1.0f, 0.0f, 0.0f,
    
        // top face (y = 0.5)
        -0.5f,  0.5f, -0.5f,  0.0f, 1.0f, 0.0f,
         0.5f,  0.5f, -0.5f,  0.0f, 1.0f, 0.0f,
         0.5f,  0.5f,  0.5f,  0.0f, 1.0f, 0.0f,
         0.5f,  0.5f,  0.5f,  0.0f, 1.0f, 0.0f,
        -0.5f,  0.5f,  0.5f,  0.0f, 1.0f, 0.0f,
        -0.5f,  0.5f, -0.5f,  0.0f, 1.0f, 0.0f,
    
        // bottom face (y = -0.5)
        -0.5f, -0.5f, -0.5f,  0.0f, -1.0f, 0.0f,
         0.5f, -0.5f, -0.5f,  0.0f, -1.0f, 0.0f,
         0.5f, -0.5f,  0.5f,  0.0f, -1.0f, 0.0f,
         0.5f, -0.5f,  0.5f,  0.0f, -1.0f, 0.0f,
        -0.5f, -0.5f,  0.5f,  0.0f, -1.0f, 0.0f,
        -0.5f, -0.5f, -0.5f,  0.0f, -1.0f, 0.0f
    };
    

    
    glm::vec3 lightPos(0.0f, 0.0f, 10.0f);
    glm::vec3 baseColor(1.0f, 1.0f, 1.0f);

    GLuint cubeVBO, cubeVAO;
    glGenVertexArrays(1, &cubeVAO);
    glGenBuffers(1, &cubeVBO);
    
    glBindVertexArray(cubeVAO);
    glBindBuffer(GL_ARRAY_BUFFER, cubeVBO);
    glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
    
    // position attribute
    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)0);
    
    // normal attribute
    glEnableVertexAttribArray(1);
    glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)(3 * sizeof(float)));
    
    glBindVertexArray(0);
    

    GLuint ssbo;
    glGenBuffers(1, &ssbo);
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, ssbo);
    glBufferData(GL_SHADER_STORAGE_BUFFER,
                instances.size() * sizeof(Instance),
                instances.data(),
                GL_STATIC_DRAW);
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, 0, ssbo);

    GLuint program = createProgram("vertex.fs", "fragment.fs");
    GLint lightPosLoc = glGetUniformLocation(program, "lightPos");
    GLint baseColorLoc = glGetUniformLocation(program, "baseColor");
    glm::vec3 cameraPos(0.0f, 0.0f, 3.0f);
    float yaw = 0.0f;
    float pitch = 0.0f;
    

    float speed = 40.0f;
    float rotSpeed = 1.5f;

    double lastTime = glfwGetTime();
    glViewport(0,0,800,600);
    glm::vec3 fogColor(237, 233, 232);
    fogColor=fogColor/255.0f;
    while (!glfwWindowShouldClose(window)) {
        double now = glfwGetTime();
        float dt = float(now - lastTime);
        lastTime = now;
        float limit = glm::radians(89.0f);
        if (pitch > limit) pitch = limit;
        if (pitch < -limit) pitch = -limit;
        
        glm::vec3 forward(
            cos(pitch) * sin(yaw),
            sin(pitch),
           -cos(pitch) * cos(yaw)
        );
        forward = glm::normalize(forward);
        

        glm::vec3 right(
            cos(yaw),
            0.0f,
            sin(yaw)
        );
        
        glm::mat4 model = glm::mat4(1.0f);
        glm::mat4 view = glm::lookAt(
            cameraPos, 
            cameraPos + forward, 
            glm::vec3(0,1,0)
        );
        glm::mat4 projection = glm::perspective(
            glm::radians(70.0f), 
            800.0f/600.0f, 
            0.1f, 
            500.0f
        );
        if (glfwGetKey(window, GLFW_KEY_W) == GLFW_PRESS)
            cameraPos += forward * speed * dt;
        if (glfwGetKey(window, GLFW_KEY_S) == GLFW_PRESS)
            cameraPos -= forward * speed * dt;
        if (glfwGetKey(window, GLFW_KEY_A) == GLFW_PRESS)
            cameraPos -= right * speed * dt;
        if (glfwGetKey(window, GLFW_KEY_D) == GLFW_PRESS)
            cameraPos += right * speed * dt;
            if (glfwGetKey(window, GLFW_KEY_LEFT) == GLFW_PRESS)
            yaw -= rotSpeed * dt;
        
        if (glfwGetKey(window, GLFW_KEY_RIGHT) == GLFW_PRESS)
            yaw += rotSpeed * dt;
        
        if (glfwGetKey(window, GLFW_KEY_UP) == GLFW_PRESS)
            pitch += rotSpeed * dt;
        
        if (glfwGetKey(window, GLFW_KEY_DOWN) == GLFW_PRESS)
            pitch -= rotSpeed * dt;
        



        glClearColor(fogColor.x,fogColor.y,fogColor.z,255);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        glUseProgram(program);
        glm::vec3 lightPos(0.0f, 0.0f, 10.0f);
        glm::vec3 cubeColor(0.74,0.74,0.74);
        
        glUniform3fv(lightPosLoc, 1, glm::value_ptr(lightPos));
        glUniform3fv(baseColorLoc, 1, glm::value_ptr(cubeColor));
        
        glUniformMatrix4fv(glGetUniformLocation(program, "model"), 1, GL_FALSE, glm::value_ptr(model));
        glUniformMatrix4fv(glGetUniformLocation(program, "view"), 1, GL_FALSE, glm::value_ptr(view));
        glUniformMatrix4fv(glGetUniformLocation(program, "projection"), 1, GL_FALSE, glm::value_ptr(projection));
        glUniform3fv(glGetUniformLocation(program, "cameraPos"), 1, glm::value_ptr(cameraPos));
        glUniform3fv(glGetUniformLocation(program, "fogColor"), 1, glm::value_ptr(fogColor)); // black fog
        
        glUniform1f(glGetUniformLocation(program, "fogStart"), 10.0f);
        glUniform1f(glGetUniformLocation(program, "fogEnd"), 200.0f);
        

        //glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);


        glBindVertexArray(cubeVAO);
        glDrawArraysInstanced(GL_TRIANGLES, 0, 36, instances.size());

        glfwSwapBuffers(window);
        glfwPollEvents();
    }

    glfwTerminate();
}
